/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.lang.ref.WeakReference;
import java.nio.ByteBuffer;
import java.util.LinkedList;
import java.util.PriorityQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.io.HeapSize;
import org.apache.hadoop.hbase.io.hfile.BlockCache;
import org.apache.hadoop.hbase.io.hfile.CachedBlock;
import org.apache.hadoop.hbase.io.hfile.CachedBlockQueue;
import org.apache.hadoop.hbase.util.ClassSize;
import org.apache.hadoop.util.StringUtils;

public class LruBlockCache
implements BlockCache,
HeapSize {
    static final Log LOG = LogFactory.getLog(LruBlockCache.class);
    static final float DEFAULT_LOAD_FACTOR = 0.75f;
    static final int DEFAULT_CONCURRENCY_LEVEL = 16;
    static final float DEFAULT_MIN_FACTOR = 0.75f;
    static final float DEFAULT_ACCEPTABLE_FACTOR = 0.85f;
    static final float DEFAULT_SINGLE_FACTOR = 0.25f;
    static final float DEFAULT_MULTI_FACTOR = 0.5f;
    static final float DEFAULT_MEMORY_FACTOR = 0.25f;
    static final int statThreadPeriod = 300;
    private final ConcurrentHashMap<String, CachedBlock> map;
    private final ReentrantLock evictionLock = new ReentrantLock(true);
    private volatile boolean evictionInProgress = false;
    private final EvictionThread evictionThread;
    private final ScheduledExecutorService scheduleThreadPool = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat("LRU Statistics #%d").build());
    private final AtomicLong size;
    private final AtomicLong elements;
    private final AtomicLong count;
    private final CacheStats stats;
    private long maxSize;
    private long blockSize;
    private float acceptableFactor;
    private float minFactor;
    private float singleFactor;
    private float multiFactor;
    private float memoryFactor;
    private long overhead;
    public static final long CACHE_FIXED_OVERHEAD = ClassSize.align(24 + 8 * ClassSize.REFERENCE + 20 + 1 + ClassSize.OBJECT);

    public LruBlockCache(long maxSize, long blockSize) {
        this(maxSize, blockSize, true);
    }

    public LruBlockCache(long maxSize, long blockSize, boolean evictionThread) {
        this(maxSize, blockSize, evictionThread, (int)Math.ceil(1.2 * (double)maxSize / (double)blockSize), 0.75f, 16, 0.75f, 0.85f, 0.25f, 0.5f, 0.25f);
    }

    public LruBlockCache(long maxSize, long blockSize, boolean evictionThread, int mapInitialSize, float mapLoadFactor, int mapConcurrencyLevel, float minFactor, float acceptableFactor, float singleFactor, float multiFactor, float memoryFactor) {
        if (singleFactor + multiFactor + memoryFactor != 1.0f) {
            throw new IllegalArgumentException("Single, multi, and memory factors  should total 1.0");
        }
        if (minFactor >= acceptableFactor) {
            throw new IllegalArgumentException("minFactor must be smaller than acceptableFactor");
        }
        if (minFactor >= 1.0f || acceptableFactor >= 1.0f) {
            throw new IllegalArgumentException("all factors must be < 1");
        }
        this.maxSize = maxSize;
        this.blockSize = blockSize;
        this.map = new ConcurrentHashMap(mapInitialSize, mapLoadFactor, mapConcurrencyLevel);
        this.minFactor = minFactor;
        this.acceptableFactor = acceptableFactor;
        this.singleFactor = singleFactor;
        this.multiFactor = multiFactor;
        this.memoryFactor = memoryFactor;
        this.stats = new CacheStats();
        this.count = new AtomicLong(0L);
        this.elements = new AtomicLong(0L);
        this.overhead = LruBlockCache.calculateOverhead(maxSize, blockSize, mapConcurrencyLevel);
        this.size = new AtomicLong(this.overhead);
        if (evictionThread) {
            this.evictionThread = new EvictionThread(this);
            this.evictionThread.start();
        } else {
            this.evictionThread = null;
        }
        this.scheduleThreadPool.scheduleAtFixedRate(new StatisticsThread(this), 300L, 300L, TimeUnit.SECONDS);
    }

    public void setMaxSize(long maxSize) {
        this.maxSize = maxSize;
        if (this.size.get() > this.acceptableSize() && !this.evictionInProgress) {
            this.runEviction();
        }
    }

    @Override
    public void cacheBlock(String blockName, ByteBuffer buf, boolean inMemory) {
        CachedBlock cb = this.map.get(blockName);
        if (cb != null) {
            throw new RuntimeException("Cached an already cached block");
        }
        cb = new CachedBlock(blockName, buf, this.count.incrementAndGet(), inMemory);
        long newSize = this.size.addAndGet(cb.heapSize());
        this.map.put(blockName, cb);
        this.elements.incrementAndGet();
        if (newSize > this.acceptableSize() && !this.evictionInProgress) {
            this.runEviction();
        }
    }

    @Override
    public void cacheBlock(String blockName, ByteBuffer buf) {
        this.cacheBlock(blockName, buf, false);
    }

    @Override
    public ByteBuffer getBlock(String blockName, boolean caching) {
        CachedBlock cb = this.map.get(blockName);
        if (cb == null) {
            this.stats.miss(caching);
            return null;
        }
        this.stats.hit(caching);
        cb.access(this.count.incrementAndGet());
        return cb.getBuffer();
    }

    protected long evictBlock(CachedBlock block) {
        this.map.remove(block.getName());
        this.size.addAndGet(-1L * block.heapSize());
        this.elements.decrementAndGet();
        this.stats.evicted();
        return block.heapSize();
    }

    private void runEviction() {
        if (this.evictionThread == null) {
            this.evict();
        } else {
            this.evictionThread.evict();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void evict() {
        if (!this.evictionLock.tryLock()) {
            return;
        }
        try {
            BlockBucket bucket;
            this.evictionInProgress = true;
            long currentSize = this.size.get();
            long bytesToFree = currentSize - this.minSize();
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Block cache LRU eviction started; Attempting to free " + StringUtils.byteDesc((long)bytesToFree) + " of total=" + StringUtils.byteDesc((long)currentSize)));
            }
            if (bytesToFree <= 0L) {
                return;
            }
            BlockBucket bucketSingle = new BlockBucket(bytesToFree, this.blockSize, this.singleSize());
            BlockBucket bucketMulti = new BlockBucket(bytesToFree, this.blockSize, this.multiSize());
            BlockBucket bucketMemory = new BlockBucket(bytesToFree, this.blockSize, this.memorySize());
            for (CachedBlock cachedBlock : this.map.values()) {
                switch (cachedBlock.getPriority()) {
                    case SINGLE: {
                        bucketSingle.add(cachedBlock);
                        break;
                    }
                    case MULTI: {
                        bucketMulti.add(cachedBlock);
                        break;
                    }
                    case MEMORY: {
                        bucketMemory.add(cachedBlock);
                    }
                }
            }
            PriorityQueue<BlockBucket> bucketQueue = new PriorityQueue<BlockBucket>(3);
            bucketQueue.add(bucketSingle);
            bucketQueue.add(bucketMulti);
            bucketQueue.add(bucketMemory);
            int remainingBuckets = 3;
            long bytesFreed = 0L;
            while ((bucket = (BlockBucket)bucketQueue.poll()) != null) {
                long overflow = bucket.overflow();
                if (overflow > 0L) {
                    long bucketBytesToFree = Math.min(overflow, (bytesToFree - bytesFreed) / (long)remainingBuckets);
                    bytesFreed += bucket.free(bucketBytesToFree);
                }
                --remainingBuckets;
            }
            if (LOG.isDebugEnabled()) {
                long single = bucketSingle.totalSize();
                long multi = bucketMulti.totalSize();
                long memory = bucketMemory.totalSize();
                LOG.debug((Object)("Block cache LRU eviction completed; freed=" + StringUtils.byteDesc((long)bytesFreed) + ", " + "total=" + StringUtils.byteDesc((long)this.size.get()) + ", " + "single=" + StringUtils.byteDesc((long)single) + ", " + "multi=" + StringUtils.byteDesc((long)multi) + ", " + "memory=" + StringUtils.byteDesc((long)memory)));
            }
        }
        finally {
            this.stats.evict();
            this.evictionInProgress = false;
            this.evictionLock.unlock();
        }
    }

    public long getMaxSize() {
        return this.maxSize;
    }

    public long getCurrentSize() {
        return this.size.get();
    }

    public long getFreeSize() {
        return this.getMaxSize() - this.getCurrentSize();
    }

    public long size() {
        return this.elements.get();
    }

    public long getEvictionCount() {
        return this.stats.getEvictionCount();
    }

    public long getEvictedCount() {
        return this.stats.getEvictedCount();
    }

    public void logStats() {
        if (!LOG.isDebugEnabled()) {
            return;
        }
        long totalSize = this.heapSize();
        long freeSize = this.maxSize - totalSize;
        LOG.debug((Object)("LRU Stats: total=" + StringUtils.byteDesc((long)totalSize) + ", " + "free=" + StringUtils.byteDesc((long)freeSize) + ", " + "max=" + StringUtils.byteDesc((long)this.maxSize) + ", " + "blocks=" + this.size() + ", " + "accesses=" + this.stats.getRequestCount() + ", " + "hits=" + this.stats.getHitCount() + ", " + "hitRatio=" + StringUtils.formatPercent((double)this.stats.getHitRatio(), (int)2) + "%, " + "cachingAccesses=" + this.stats.getRequestCachingCount() + ", " + "cachingHits=" + this.stats.getHitCachingCount() + ", " + "cachingHitsRatio=" + StringUtils.formatPercent((double)this.stats.getHitCachingRatio(), (int)2) + "%, " + "evictions=" + this.stats.getEvictionCount() + ", " + "evicted=" + this.stats.getEvictedCount() + ", " + "evictedPerRun=" + this.stats.evictedPerEviction()));
    }

    public CacheStats getStats() {
        return this.stats;
    }

    @Override
    public long heapSize() {
        return this.getCurrentSize();
    }

    public static long calculateOverhead(long maxSize, long blockSize, int concurrency) {
        return CACHE_FIXED_OVERHEAD + (long)ClassSize.CONCURRENT_HASHMAP + (long)Math.ceil((double)maxSize * 1.2 / (double)blockSize) * (long)ClassSize.CONCURRENT_HASHMAP_ENTRY + (long)(concurrency * ClassSize.CONCURRENT_HASHMAP_SEGMENT);
    }

    private long acceptableSize() {
        return (long)Math.floor((float)this.maxSize * this.acceptableFactor);
    }

    private long minSize() {
        return (long)Math.floor((float)this.maxSize * this.minFactor);
    }

    private long singleSize() {
        return (long)Math.floor((float)this.maxSize * this.singleFactor * this.minFactor);
    }

    private long multiSize() {
        return (long)Math.floor((float)this.maxSize * this.multiFactor * this.minFactor);
    }

    private long memorySize() {
        return (long)Math.floor((float)this.maxSize * this.memoryFactor * this.minFactor);
    }

    @Override
    public void shutdown() {
        this.scheduleThreadPool.shutdown();
    }

    public static class CacheStats {
        private final AtomicLong hitCount = new AtomicLong(0L);
        private final AtomicLong hitCachingCount = new AtomicLong(0L);
        private final AtomicLong missCount = new AtomicLong(0L);
        private final AtomicLong missCachingCount = new AtomicLong(0L);
        private final AtomicLong evictionCount = new AtomicLong(0L);
        private final AtomicLong evictedCount = new AtomicLong(0L);

        public void miss(boolean caching) {
            this.missCount.incrementAndGet();
            if (caching) {
                this.missCachingCount.incrementAndGet();
            }
        }

        public void hit(boolean caching) {
            this.hitCount.incrementAndGet();
            if (caching) {
                this.hitCachingCount.incrementAndGet();
            }
        }

        public void evict() {
            this.evictionCount.incrementAndGet();
        }

        public void evicted() {
            this.evictedCount.incrementAndGet();
        }

        public long getRequestCount() {
            return this.getHitCount() + this.getMissCount();
        }

        public long getRequestCachingCount() {
            return this.getHitCachingCount() + this.getMissCachingCount();
        }

        public long getMissCount() {
            return this.missCount.get();
        }

        public long getMissCachingCount() {
            return this.missCachingCount.get();
        }

        public long getHitCount() {
            return this.hitCount.get();
        }

        public long getHitCachingCount() {
            return this.hitCachingCount.get();
        }

        public long getEvictionCount() {
            return this.evictionCount.get();
        }

        public long getEvictedCount() {
            return this.evictedCount.get();
        }

        public double getHitRatio() {
            return (float)this.getHitCount() / (float)this.getRequestCount();
        }

        public double getHitCachingRatio() {
            return (float)this.getHitCachingCount() / (float)this.getRequestCachingCount();
        }

        public double getMissRatio() {
            return (float)this.getMissCount() / (float)this.getRequestCount();
        }

        public double getMissCachingRatio() {
            return (float)this.getMissCachingCount() / (float)this.getRequestCachingCount();
        }

        public double evictedPerEviction() {
            return (float)this.getEvictedCount() / (float)this.getEvictionCount();
        }
    }

    static class StatisticsThread
    extends Thread {
        LruBlockCache lru;

        public StatisticsThread(LruBlockCache lru) {
            super("LruBlockCache.StatisticsThread");
            this.setDaemon(true);
            this.lru = lru;
        }

        @Override
        public void run() {
            this.lru.logStats();
        }
    }

    private static class EvictionThread
    extends Thread {
        private WeakReference<LruBlockCache> cache;

        public EvictionThread(LruBlockCache cache) {
            super("LruBlockCache.EvictionThread");
            this.setDaemon(true);
            this.cache = new WeakReference<LruBlockCache>(cache);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            while (true) {
                EvictionThread evictionThread = this;
                synchronized (evictionThread) {
                    try {
                        this.wait();
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                }
                LruBlockCache cache = (LruBlockCache)this.cache.get();
                if (cache == null) break;
                cache.evict();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void evict() {
            EvictionThread evictionThread = this;
            synchronized (evictionThread) {
                this.notify();
            }
        }
    }

    private class BlockBucket
    implements Comparable<BlockBucket> {
        private CachedBlockQueue queue;
        private long totalSize = 0L;
        private long bucketSize;

        public BlockBucket(long bytesToFree, long blockSize, long bucketSize) {
            this.bucketSize = bucketSize;
            this.queue = new CachedBlockQueue(bytesToFree, blockSize);
            this.totalSize = 0L;
        }

        public void add(CachedBlock block) {
            this.totalSize += block.heapSize();
            this.queue.add(block);
        }

        public long free(long toFree) {
            LinkedList<CachedBlock> blocks = this.queue.get();
            long freedBytes = 0L;
            for (CachedBlock cb : blocks) {
                if ((freedBytes += LruBlockCache.this.evictBlock(cb)) < toFree) continue;
                return freedBytes;
            }
            return freedBytes;
        }

        public long overflow() {
            return this.totalSize - this.bucketSize;
        }

        public long totalSize() {
            return this.totalSize;
        }

        @Override
        public int compareTo(BlockBucket that) {
            if (this.overflow() == that.overflow()) {
                return 0;
            }
            return this.overflow() > that.overflow() ? 1 : -1;
        }
    }
}

